/*
Test out low-level (canonicalization array)
FEM symmetry interface.

Orion Sky Lawlor, olawlor@acm.org, 7/25/2002
*/
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "charm++.h"
#include "tcharmc.h"
#include "fem.h"
#include "netfem.h"

//Bits to indicate symmetries:
#define LEFT_BOUND 4
#define RIGHT_BOUND 16

//Number of time steps to simulate
int tsteps=10;
const int dim=50;//Elements per side of the FEM mesh
const int np=4; //Nodes per element for a quad

//Index of a node at x,y
int nodeDex(int x,int y) {return x+y*(dim+1);}
//Index of element connecting x,y to x+1,y+1
int elemDex(int x,int y) {return x+y*dim;}

extern "C" void
init(void)
{
  CkPrintf("init called\n");
  int nelems=dim*dim, nnodes=(dim+1)*(dim+1);
  
  //Describe the nodes and elements
  FEM_Set_node(nnodes,2);
  FEM_Set_elem(0,nelems,1,np);
  
  int x,y,e,n;
  //Create element & node data
  double *elements=new double[nelems];
  for (e=0;e<nelems;e++) elements[e]=10.0*(rand()%1000)/1000.0;
  FEM_Set_elem_data(0,elements);
  delete[] elements;
  
  double *nodes=new double[2*nnodes];
  for(y=0;y<dim+1;y++) for (x=0;x<dim+1;x++) {
    nodes[nodeDex(x,y)*2+0]=x/float(dim);
    nodes[nodeDex(x,y)*2+1]=y/float(dim);
  }
  FEM_Set_node_data(nodes);
  delete[] nodes;
  
  //Create the connectivity array
  int *conn=new int[dim*dim*np];
  for(y=0;y<dim;y++) for (x=0;x<dim;x++) {
  	   e=elemDex(x,y);
	   conn[e*np+0]=nodeDex(x  ,y  );
	   conn[e*np+1]=nodeDex(x  ,y+1);
	   conn[e*np+2]=nodeDex(x+1,y+1);
	   conn[e*np+3]=nodeDex(x+1,y  );
  }
  FEM_Set_elem_conn(0,conn);
  delete[] conn;

#if 1
//Set up symmetries: left and right boundaries match up
  int *canon=new int[nnodes];
  int *sym=new int[nnodes];
  for (n=0;n<nnodes;n++) {canon[n]=n; sym[n]=0;}
  for (y=0;y<dim+1;y++) {
    canon[nodeDex(dim,y)]=nodeDex(0,y); //Paste right boundary on left    
    sym[nodeDex(0  ,y)]+=LEFT_BOUND; //Mark left boundary
    sym[nodeDex(dim,y)]+=RIGHT_BOUND; //Mark right boundary
  }
  FEM_Set_sym_nodes(canon,sym);
  delete[] canon;
  delete[] sym;
#endif

#if 1
//Set up ghost layers:
  if (0) 
  { /*Match across single nodes*/
     static const int quad2node[]={0,1,2,3};
     FEM_Add_ghost_layer(1,1);
     FEM_Add_ghost_elem(0,4,quad2node);
  } else if (1) 
  { /*Match edges*/
#if 1 /*Include ignored "-1" nodes as a test*/
     static const int quad2edge[]= {0,1,-1,  1,2,-1,  2,3,-1,  3,0,-1};
     FEM_Add_ghost_layer(3,1);
     FEM_Add_ghost_elem(0,4,quad2edge);
#else
     static const int quad2edge[]= {0,1,  1,2,  2,3,  3,0};
     FEM_Add_ghost_layer(2,1);
     FEM_Add_ghost_elem(0,4,quad2edge);
#endif
/*Add a second layer
     FEM_Add_ghost_layer(2,0);
     FEM_Add_ghost_elem(0,4,quad2edge);
*/
  }
#endif

}

extern "C" void
driver(void)
{
  int nnodes,nelems,nnodeData,nelemData,np;
  int ngnodes, ngelems; //Counts including ghosts

  FEM_Print("Starting driver...");
  int myId = FEM_My_partition();
  FEM_Print_partition();

  FEM_Get_node(&nnodes,&nnodeData);
  double *nodeData=new double[nnodeData*nnodes];
  FEM_Get_node_data(nodeData);  

  FEM_Get_elem(0,&nelems,&nelemData,&np);
  int *conn=new int[np*nelems];
  FEM_Get_elem_conn(0,conn);
  double *elData=new double[nelemData*nelems];
  FEM_Get_elem_data(0,elData);
  
  int *elSym=new int[nelems];
  int *noSym=new int[nnodes];
  FEM_Get_sym(0,elSym);
  FEM_Get_sym(-1,noSym);

  double *elSymVal=new double[nelems];
  double *noSymVal=new double[nnodes];
  double *elGhost=new double[nelems];
  double *noGhost=new double[nnodes];
  
//Clip off ghost nodes/elements
  ngnodes=nnodes; ngelems=nelems;
  nnodes=FEM_Get_node_ghost();
  nelems=FEM_Get_elem_ghost(0);
  
  int e,n;
  for (n=0;n<ngnodes;n++)
  {
    noSymVal[n]=noSym[n];
    noGhost[n]=n>=nnodes;
    double scl=10.0;
    double dx=scl*(0.01*(myId%3)-0.001*(myId/6));
    double dy=scl*(0.01*(myId/3)-0.001*(myId%6));
    if (noSym[n]&LEFT_BOUND) dx-=1.0; //Shift node over to left boundary
    if (noSym[n]&RIGHT_BOUND) dx+=1.0; //Shift over to right boundary   
    nodeData[2*n+0]+=dx;
    nodeData[2*n+1]+=dy;
    if (noSym[n]) CkPrintf("[%d] Node %d has symmetry %d\n",myId,n,noSym[n]);
  }
  for (e=0;e<ngelems;e++) {
    elSymVal[e]=elSym[e];
    elGhost[e]=e>=nelems;
    if (elSym[e]) CkPrintf("[%d] El %d has symmetry %d\n",myId,e,elSym[e]);
  }
  

  int doubleField=FEM_Create_simple_field(FEM_DOUBLE,1);
#if 1
  for (int e=nelems;e<ngelems;e++) elData[e]=-1.0;
  FEM_Update_ghost_field(doubleField,0,elData);
  for (int e=nelems;e<ngelems;e++) {
    if (elData[e]==-1.0) {
      CkPrintf("[%d] Ghost field update failed for %d",myId,e);
    }
  }
#endif
  int ts=0;
  while(1) {
    NetFEM f=NetFEM_Begin(FEM_My_partition(),ts,2,NetFEM_POINTAT);
    NetFEM_Nodes(f,ngnodes,nodeData,"Node Locs");
    NetFEM_Scalar(f,noSymVal,1,"Symmetries");
    NetFEM_Scalar(f,noGhost,1,"Ghost");
    NetFEM_Elements(f,ngelems,np,conn,"Elements");
    NetFEM_Scalar(f,elData,1,"Random");
    NetFEM_Scalar(f,elSymVal,1,"Symmetries");
    NetFEM_Scalar(f,elGhost,1,"Ghost");
    NetFEM_End(f);
    ts++;
    FEM_Barrier();
  }
  
}

extern "C" void
mesh_updated(int param)
{
  CkPrintf("mesh_updated(%d) called.\n",param);
}
