/*
Returns a set of views, given a viewpoint.  
This version requests new views over CCS.

Orion Sky Lawlor, olawlor@acm.org, 8/29/2002
*/
#include <math.h>
#include <stdio.h>
#include "sixty.h"
#include "ccs-client.h"
#include "pup.h"
#include "ckviewpoint.h"
#include "ckviewable.h"
#include "pup_toNetwork4.h"
#include "porthread.h"


// double startTime=CmiWallTimer();

void callServiceIncoming(void *forObj);
void callServiceOutgoing(void *forObj);

class ccsViewSource : public viewSource {

/***** Shared state: *****/
private:
	int clientID; //My identifier
	
	void status(const char *what,int prio=1) {
		// if (prio>=2)
		// 	printf("(%.3f s) CCS Client status> %s\n",CmiWallTimer()-startTime,what);
	}
	
/**** Incoming images, from server ****/
private:
	//Incoming server control:
	CcsServer responses; //Incoming server: receive CkView's
	
	//Incoming Interface:
	//Written by incoming thread; read by display thread
	// NULL-> nothing available.
	volatile char *incomingData; 
	
	//Ask for the latest set of 3d views:
	void askForResponses(void) {
		status("Asking for responses");
		char buf[32];
		PUP_toNetwork4_pack p(buf);
		p|clientID;
		CcsSendRequest(&responses,"lv3d_getViews",0,p.size(),buf);
	}
	//Grab the latest set of responses off the network
	char *waitForResponses(void) {
		int retSiz=0; void *retBuf=NULL;
		const int timeout=8*60*60; //Wait up to 8 hours
		CcsRecvResponseMsg(&responses,&retSiz,&retBuf,timeout);
		return (char *)retBuf;
	}
	
public:
	void serviceIncoming(void) {
		while (1) { //Network service loop:
			askForResponses();
			incomingData=waitForResponses();
			while (incomingData!=NULL) 
				porthread_yield(10); //Wait until data is read by other thread
			
		}
	}

/**** Outgoing viewpoints, to server ****/
private:
	//Outgoing server: send out CkViewpoint's
	CcsServer requests; 
	
	//Outgoing interface:
	volatile bool reqUpdated; //Set to true by display, set to false by outgoing
	CkViewpoint req;

public:
	void serviceOutgoing(void) {
		while (1) { //Network service loop:
			while (reqUpdated==false) 
				porthread_yield(10); //Wait until new viewpoint is given by display thread
			CkViewpoint oreq=req;
			reqUpdated=false;
			
			/*Send the new viewpoint off to processor 0:*/
			const int bufLen=2*sizeof(oreq);
			char buf[bufLen];
			PUP_toNetwork4_pack p(buf);
			p|oreq;
			status("   Sending off new viewpoint");
			CcsSendRequest(&requests,"lv3d_newViewpoint",0,p.size(),buf);
			CcsRecvResponse(&requests,bufLen,buf,60); //response is empty
		}
	}

	

/***** Display thread: *****/
private:
	
	viewDest *dest; //Place to send responses
	
	//Grab responses (CkViews, in blocks) off the network thread's buffer
	bool probeResponses(void) {
		if (0==incomingData) return false; //Nothing to read
		
		status("Grabbing textures from network");
		char *retBuf=(char *)incomingData;
		incomingData=NULL;
		if (retBuf!=NULL) {
			PUP_toNetwork4_unpack p(retBuf);
			int nViews=0;
			p|nViews;
			for (int i=0;i<nViews;i++) {
				CkView *v=pup_unpack(p);
				dest->viewResponse(v);
				v->unref();
				status("   Uploading new texture",3);
			}
			free(retBuf);
		}
		return true;
	}
	
public:
	ccsViewSource(int argc, char *argv[]) {
		if (argc<2) {
			printf("Usage: %s <ccs server name> <ccs server port>\n",argv[0]);
			exit(1);
		}
		const char *hostName=argv[1];
		int port=atoi(argv[2]);
		printf("Connecting to CCS server %s:%d\n",hostName,port);
		CcsConnect(&requests,hostName,port,NULL);
		CcsConnect(&responses,hostName,port,NULL);
		status("connected");
		clientID=1234; //FIXME: should ask server which client ID to use.
		
		incomingData=NULL;
		porthread_create(callServiceIncoming,this);
		
		// req.clientID=clientID;
		reqUpdated=false;
		porthread_create(callServiceOutgoing,this);
	}
	
	virtual void viewRequest(viewDest *dest_,const CkViewpoint &vp) {
		dest=dest_;
		req=vp; //FIXME: thread-unsafe write
		reqUpdated=true;
		
		probeResponses();
	}
	
	virtual void viewPoll(void) {
		if (!probeResponses()) 
			porthread_yield(10);
		//printf("Idle.\n");
	}
};

void callServiceIncoming(void *forObj) {
	ccsViewSource *vs=(ccsViewSource *)forObj;
	vs->serviceIncoming();
}
void callServiceOutgoing(void *forObj) {
	ccsViewSource *vs=(ccsViewSource *)forObj;
	vs->serviceOutgoing();
}

viewSource *createViewSource(int argc,char *argv[])
{
	return new ccsViewSource(argc,argv);
}

