package projector.core;

import java.io.*;
import java.util.*;

/**
 * Author: Arun Singla
 * May 2, 2002
 */

/**
 * TraceReader
 * A generic reader for projector log files.
 */
public class TraceReader
{
	// public attributes *****************************

	// public methods ********************************
	/**
	 * constructor: TraceReader(String ptcFName)
	 * args: path - full path name of log files directory
	 *     	 pgm  - name of the program
	 */
	public TraceReader(String path, String pgm) throws AbortException
	{
		this.path = path;
		this.pgm  = pgm;
		pgmPath   = path+pgm;

		ptcReader = new PTCReader(path+pgm+".ptc");
		
		openLogFiles();

		globalEventFlag = false;
		nextLang = nextSeek = 0;
		nextSame = false;
	}

	/**
	 * void close()
	 * - deallocates the resources used by projector
	 */
	public void close() throws AbortException
	{
		closeLogFiles();
	}

	/**
	 * void reset(int language, int proc)
	 * - restarts the reading of log file 'pgm.proc.lang.log' to beginging
	 */
	public void reset(int language, int proc) throws AbortException
	{
	//	Debug.print("TraceReader:reset: args -- language = " +language+ ", proc = " +proc);
		try{
		// close the corresponding input stream
		if(lStream[language][proc] != null)
			lStream[language][proc].close();

		// open the input stream again
		if(ptcReader.lName[language] == null)
			throw new AbortException("TraceReader:reset: Invalid argument language");
		String s = ptcReader.lName[language] + ".log";
		lStream[language][proc] = new BufferedReader(new FileReader(pgmPath+"."+proc+"."+s));
		// remove the useful-for-humans-only header
		lStream[language][proc].readLine();
		}
		catch (ArrayIndexOutOfBoundsException e) {
			throw new AbortException("TraceReader:reset:ArrayIndexOutOfBoundsException");
		}
		catch (FileNotFoundException e) {
			throw new AbortException("TraceReader:reset:FileNotFoundException");
		}
		catch(IOException e) {
			throw new AbortException("TraceReader:reset:IOException");
		}
	}

	/**
	 *resets reading of all logfiles for processor proc
	 */
	public void resetAll(int proc){
		int i=0;
		try{
		for(i=0;i<ptcReader.numLangs;i++){
			reset(ptcReader.langIDs[i],proc);
		}
		}catch(Exception e){
   			System.out.println("Unable to reset language " + i+ "for processor " + proc);
		}
	};

    /**
	 * boolean nextEvent(EventObject obj)
	 * - EventObj.proc specifies the processor number
	 * - EventObj.language specifies the language ID of the event requested
	 * - return value indicates found or not
	 */
 	public boolean nextEvent(EventObject obj) throws AbortException
	{
		if(obj.language == 0) {
			throw new AbortException("TraceReader:nextEvent(EventObject): 0 is invalid language ID");
		}
		else {
		  try{
		  //debug output
		  String log = lStream[obj.language][obj.proc].readLine();
		  if(log == null) // end of stream
		  	return false;
		  StringTokenizer st = new StringTokenizer(log, " ");
		  obj.eventType = Integer.parseInt(st.nextToken());
		  obj.timestamp = (new Double(st.nextToken()).doubleValue());
		  // eliminate seek values
		  if(Integer.parseInt(st.nextToken())!=0)	// prev seek
		  	st.nextToken();
		  if(Integer.parseInt(st.nextToken())!=0)	// next seek
		  	st.nextToken();
		  obj.eLen = Integer.parseInt(st.nextToken());	// number of integers in entity
		  if(obj.eLen != 0) {
		  	obj.entity = new int[obj.eLen];
			for(int i=0; i<obj.eLen; i++) {
				obj.entity[i] = Integer.parseInt(st.nextToken());
			}
		  }
		  obj.iLen = Integer.parseInt(st.nextToken());	// number of integers in IData
		  if(obj.iLen != 0) {
		  	 obj.iData = new int[obj.iLen];
			for(int i=0; i<obj.iLen; i++) {
				obj.iData[i] = Integer.parseInt(st.nextToken());
			}
	//		obj.parseIData(iData);
		  }
		  if(st.hasMoreTokens()) {
				obj.sData = new String(st.nextToken());
		  }
		  }
		  catch(IOException e) {
			throw new AbortException("TraceReader:nextEvent(EventObject):IOException");
		  }
		  catch(NumberFormatException e) {
			throw new AbortException("TraceReader:nextEvent(EventObject):NumberFormatException");
		  }
		}
		return true;
	}

    /**
	 * boolean nextEvent(EventObject obj, int eventType)
	 * - EventObj.proc specifies the processor number
	 * - EventObject.language specifies the language ID of the event requested
	 *   of the type 'eventType'
	 * - return value indicates found or not
	 */
	public boolean nextEvent(EventObject obj, int eventType) throws AbortException
	{
		boolean found = false;
		while(found = nextEvent(obj)) {
		  if (obj.eventType == eventType)
			return true;
		}
		return false;	// not found; EOF reached
	}

    /**
	 * boolean nextEvent(EventObject obj, double timestamp)
	 * - EventObj.proc specifies the processor number
	 * - EventObject.language specifies the language ID of the event requested
	 *   which occured after 'timestamp'
	 * - return value indicates found or not
	 *
	 * NOTE: it is NOT the next event after timestamp from the begining of log
	 */
	public boolean nextEvent(EventObject obj, double timestamp) throws AbortException
	{
		boolean found = false;
		while(found = nextEvent(obj)) {
		  if (obj.timestamp >= timestamp)
			return true;
		}
		return false;	// not found; EOF reached
	}

    /**
	 * boolean nextGlobalEvent(EventObject obj)
	 * - EventObj.proc specifies the processor number
	 * - EventObj.language specifies the language ID of the FIRST event
	 *   NOTE: EventObj.language is set only to initialize
	 * - return value indicates found or not
	 */
	 int language;
 	public boolean nextGlobalEvent(EventObject obj) throws AbortException
	{
		int proc = obj.proc;


		if(globalEventFlag == false) {
			globalEventFlag = true;
			language = obj.language;
		}
		else {

			language = nextLang;

	  		try{
			 	long skip = nextSeek - currSeek[language][proc];
				//System.out.println("skipping " + skip+ " to " + nextSeek);
				gStream[language][proc].skip(skip);
			} catch(Exception e) {
				if(e instanceof IOException){
		  		throw new AbortException("TraceReader:nextGlobalEvent(EventObject):IOException");
				}else{
					System.out.println(language+" "+proc+" "+currSeek.length);
					System.exit(1);
				}	
			}
			currSeek[language][proc] = nextSeek;
		}

	  	try{

		String log = gStream[language][proc].readLine();

		if(log == null) // end of stream
			return false;
		currSeek[language][proc] += (log.length()+1);
		StringTokenizer st = new StringTokenizer(log, " ");
		obj.eventType = Integer.parseInt(st.nextToken());

		obj.timestamp = (new Double(st.nextToken()).doubleValue());
		obj.language = language;

		if(Integer.parseInt(st.nextToken())!=0)	// prev seek
			st.nextToken();


		nextLang = Integer.parseInt(st.nextToken());	// next language ID
		nextSame = (nextLang==0)?true:false;
		if(nextSame){
			nextLang = language;

			nextSeek = (int )currSeek[language][proc];
		}
		else{
			nextSeek = Integer.parseInt(st.nextToken());	// next seek
   			//System.out.println("Leaving Language " + language + " next Lang " + nextLang);
			//System.out.println("nextSeek "+nextSeek+"nextLang " + nextLang);
		}

		obj.eLen = Integer.parseInt(st.nextToken());	// number of integers in entity

		if(obj.eLen != 0) {
		  obj.entity = new int[obj.eLen];
		  for(int i=0; i<obj.eLen; i++) {
		  	obj.entity[i] = Integer.parseInt(st.nextToken());
		  }
		  //obj.parseEntity(entity);
		}
		obj.iLen = Integer.parseInt(st.nextToken());	// number of integers in IData
		if(obj.iLen != 0) {
		  obj.iData = new int[obj.iLen];
		  for(int i=0; i<obj.iLen; i++) {
		  	obj.iData[i] = Integer.parseInt(st.nextToken());
		  }
		  //obj.parseIData(iData);
		}
		if(st.hasMoreTokens()) {
		  obj.sData = new String(st.nextToken());
		}
		}
		catch(IOException e) {
		  throw new AbortException("TraceReader:nextGlobalEvent(EventObject):IOException");
		}
		catch(NumberFormatException e) {
		  throw new AbortException("TraceReader:nextGlobalEvent(EventObject):NumberFormatException");
		}

		return true;
	}

    /**
	 * boolean nextGlobalEvent(EventObject obj, int eventType)
	 * - EventObj.proc specifies the processor number
	 * - return value indicates found or not
	 */
	public boolean nextGlobalEvent(EventObject obj, int eventType) throws AbortException
	{
		boolean found = false;
		while(found = nextGlobalEvent(obj)) {
		  if (obj.eventType == eventType)
			return true;
		}
		return false;	// not found; EOF reached
	}

    /**
	 * boolean nextGlobalEvent(EventObject obj, double timestamp)
	 * - EventObj.proc specifies the processor number
	 * - return value indicates found or not
	 *
	 * NOTE: it is NOT the next event after timestamp from the begining of log
	 */
	public boolean nextGlobalEvent(EventObject obj, double timestamp) throws AbortException
	{
		boolean found = false;
		//System.out.println("Start Time = " + timestamp);
		while(found = nextGlobalEvent(obj)) {
		  if (obj.timestamp >= timestamp)
			return true;
		}
		return false;	// not found; EOF reached
	}

	// private attributes *****************************
	private String path;	// absolute path of log file directory
	private String pgm;		// name of program
	private String pgmPath;	// == path + pgm
	private PTCReader ptcReader;
	private BufferedReader[][] lStream;	// language specific stream

	// private attributes for keeping track of current global event
	// while using 'nextGlobalEvent' method
	private BufferedReader[][] gStream; 	// global streams
	private long[][]           currSeek;	// current seek value in 'gStream'
	private boolean globalEventFlag;
	private boolean nextSame;
	private int     nextLang;
	private int     nextSeek;

	// private methods ********************************
	/** void openLogFiles()
	  * - creates BufferedReader Stream for all log files
	  * - the streams are referenced in 'lStream' and 'gStream'
	  */
	private void openLogFiles() throws AbortException
	{
		String s="";
		//Debug.print("TraceReader:openLogFiles ...");
		lStream = new BufferedReader[ptcReader.maxLangs][ptcReader.numProcs];
		gStream = new BufferedReader[ptcReader.maxLangs][ptcReader.numProcs];
		currSeek = new long[ptcReader.maxLangs][ptcReader.numProcs];
		int i=0,j=0;
		try {
		for( i=0; i<ptcReader.maxLangs; i++)	{
		  if(ptcReader.lName[i] == null) continue;
		  s = ptcReader.lName[i] + ".log";
		  for(j=0; j<ptcReader.numProcs; j++)	{
			lStream[i][j] = new BufferedReader(new FileReader(pgmPath+"."+j+"."+s));
			gStream[i][j] = new BufferedReader(new FileReader(pgmPath+"."+j+"."+s));
			// remove the useful-for-humans-only header
			String firstLine = lStream[i][j].readLine();
			firstLine = gStream[i][j].readLine();
			currSeek[i][j] = firstLine.length()+1;
		  }
		}
		}
		catch (ArrayIndexOutOfBoundsException e) {
			System.out.println("array out of index\n");
			throw new AbortException("TraceReader:openLogFiles:ArrayIndexOutOfBoundsException");
			
		}
		catch (FileNotFoundException e) {
			System.out.println("filenot found:["+pgmPath+"."+j+"."+s+"]\n");
			throw new AbortException("TraceReader:openLogFiles:FileNotFoundException");
			
		}
		catch (IOException e) {
			System.out.println("IOexception found\n");
			throw new AbortException("TraceReader:openLogFiles:IOException");
		}

		globalEventFlag = false;
		nextLang = nextSeek = 0;
		nextSame = false;
	}

	/** void closeLogFiles()
	  * - closes BufferedReader Stream for all open log files
	  */
	private void closeLogFiles() throws AbortException
	{
		//Debug.print("TraceReader:closeLogFiles ...");
		for(int i=0; i<ptcReader.maxLangs; i++)	{
		  for(int j=0; j<ptcReader.numProcs; j++)	{
		  	try{
			if(lStream[i][j] != null) lStream[i][j].close();
			if(gStream[i][j] != null) gStream[i][j].close();
			currSeek[i][j] = 0;
			}
			catch(IOException e) {
				throw new AbortException("TraceReader:closeLogFiles:IOException");
			}
		  }
		}
	}

	public int getMaxLangs(){
		return ptcReader.maxLangs;
	}
	public String [] getLanguageNames(){
		return ptcReader.lName;
	}
}

